# -*- coding: utf-8 -*-
"""Ridge.ipynb

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/1L6zxwTxw8jfjYBA8dOAdpSpHp6mqHrq-
"""

import numpy as np
import matplotlib.pyplot as plt
from sklearn.linear_model import LinearRegression, Ridge
from sklearn.model_selection import train_test_split
from sklearn.metrics import mean_squared_error, r2_score
from sklearn.preprocessing import StandardScaler

# 1. Generate some synthetic data
# We'll create data with some multicollinearity to see Ridge's benefits
np.random.seed(42)
n_samples = 100
n_features = 5

# Create somewhat correlated features
X = np.random.rand(n_samples, n_features)
X[:, 1] = X[:, 0] * 0.8 + np.random.rand(n_samples) * 0.2  # Feature 1 correlated with Feature 0
X[:, 2] = X[:, 0] * 0.5 + X[:, 3] * 0.5 + np.random.rand(n_samples) * 0.1 # Feature 2 depends on 0 and 3

# True coefficients
true_coefficients = np.array([2.5, -1.0, 3.0, 0.5, -2.0])
y = X @ true_coefficients + np.random.randn(n_samples) * 0.5 # Add some noise

# Split data into training and testing sets
X_train, X_test, y_train, y_test = train_test_split(X, y, test_size=0.3, random_state=42)

# It's generally good practice to scale features for Ridge Regression
# as the regularization term is sensitive to the scale of the features.
scaler = StandardScaler()
X_train_scaled = scaler.fit_transform(X_train)
X_test_scaled = scaler.transform(X_test)

# 2. Apply Ridge Regression

# We'll try different alpha values (regularization strength)
# A larger alpha means stronger regularization.
alphas = [0.01, 0.1, 1.0, 10.0, 100.0]

print("--- Ridge Regression ---")
for alpha in alphas:
    ridge_model = Ridge(alpha=alpha)
    ridge_model.fit(X_train_scaled, y_train)
    y_pred_ridge = ridge_model.predict(X_test_scaled)

    mse_ridge = mean_squared_error(y_test, y_pred_ridge)
    r2_ridge = r2_score(y_test, y_pred_ridge)

    print(f"\nAlpha: {alpha}")
    print(f"Ridge Coefficients: {ridge_model.coef_}")
    print(f"Ridge Intercept: {ridge_model.intercept_}")
    print(f"Ridge Mean Squared Error: {mse_ridge:.4f}")
    print(f"Ridge R-squared: {r2_ridge:.4f}")

# 3. Compare with standard Linear Regression (OLS)
print("\n--- Linear Regression (OLS) ---")
linear_model = LinearRegression()
linear_model.fit(X_train_scaled, y_train)
y_pred_linear = linear_model.predict(X_test_scaled)

mse_linear = mean_squared_error(y_test, y_pred_linear)
r2_linear = r2_score(y_test, y_pred_linear)

print(f"Linear Regression Coefficients: {linear_model.coef_}")
print(f"Linear Regression Intercept: {linear_model.intercept_}")
print(f"Linear Regression Mean Squared Error: {mse_linear:.4f}")
print(f"Linear Regression R-squared: {r2_linear:.4f}")

# Optional: Visualize the coefficients as a function of alpha (regularization strength)
# This helps understand how coefficients shrink with increasing alpha.
n_alphas = 200
alphas_plot = np.logspace(-5, 2, n_alphas)
coefs = []

for a in alphas_plot:
    ridge = Ridge(alpha=a, fit_intercept=True)
    ridge.fit(X_train_scaled, y_train)
    coefs.append(ridge.coef_)

plt.figure(figsize=(10, 6))
ax = plt.gca()
ax.plot(alphas_plot, coefs)
ax.set_xscale("log")
ax.set_xlim(ax.get_xlim()[::-1])  # Reverse axis
plt.xlabel("alpha (regularization strength)")
plt.ylabel("Coefficients")
plt.title("Ridge Coefficients as a Function of Regularization")
plt.legend([f"Feature {i}" for i in range(n_features)])
plt.axis("tight")
plt.show()